<?php
/**
 * ارسال پیامک با ملی‌پیامک
 * - روش ۱: Send.asmx (SendSimpleSMS2) — خودمان متن و کد ۴ رقمی می‌سازیم
 * - روش ۲: کنسول OTP — درخواست به آدرس OTP، کد از پاسخ
 */
if (!defined('NOVATAN_API')) {
    define('NOVATAN_API', true);
}
require_once __DIR__ . '/config.php';

/**
 * ارسال پیامک با وب‌سرویس Send.asmx (SendSimpleSMS2)
 * پارامترها مطابق WSDL: username, password, to, from, text, isflash
 * @return array { success: bool, message?: string }
 */
function send_simple_sms(string $to, string $text): array {
    $username = defined('SMS_USERNAME') ? (string) SMS_USERNAME : '';
    $password = defined('SMS_PASSWORD') ? (string) SMS_PASSWORD : '';
    $from     = defined('SMS_FROM')     ? (string) SMS_FROM     : '';

    if ($from === '' || $password === '') {
        return ['success' => false, 'message' => 'تنظیمات پیامک (شماره خط یا رمز) پر نشده است.'];
    }

    $url = 'https://api.payamak-panel.com/post/Send.asmx/SendSimpleSMS2';
    $body = http_build_query([
        'username' => $username,
        'password' => $password,
        'to'       => $to,
        'from'     => $from,
        'text'     => $text,
        'isflash'  => 'false',
    ]);

    $ch = curl_init($url);
    curl_setopt($ch, CURLOPT_POST, true);
    curl_setopt($ch, CURLOPT_POSTFIELDS, $body);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_HTTPHEADER, ['Content-Type: application/x-www-form-urlencoded']);
    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
    curl_setopt($ch, CURLOPT_TIMEOUT, 15);
    $result = curl_exec($ch);
    $errNo = curl_errno($ch);
    curl_close($ch);

    if ($errNo) {
        return ['success' => false, 'message' => 'خطا در اتصال به سرویس پیامک.'];
    }

    // پاسخ XML است؛ مقدار داخل <string>...</string> یا مشابه
    $num = trim(strip_tags($result));
    $num = preg_replace('/\s+/', '', $num);
    $n = is_numeric($num) ? (int) $num : 0;
    if ($n > 0) {
        return ['success' => true];
    }
    $errors = [
        0  => 'نام کاربری یا رمز اشتباه است',
        2  => 'اعتبار کافی نیست',
        5  => 'شماره فرستنده معتبر نیست',
        18 => 'شماره گیرنده نامعتبر است',
    ];
    return ['success' => false, 'message' => $errors[$n] ?? ('خطای ارسال: ' . $num)];
}

/**
 * ارسال کد OTP به شماره (کنسول ملی‌پیامک – درخواست به آدرس OTP)
 */
function send_otp(string $to): array {
    $url = defined('SMS_OTP_API_URL') ? trim((string) SMS_OTP_API_URL) : '';
    if ($url === '') {
        return ['success' => false, 'message' => 'آدرس OTP تنظیم نشده است.'];
    }

    $dataString = json_encode(['to' => $to]);
    $ch = curl_init($url);
    curl_setopt($ch, CURLOPT_CUSTOMREQUEST, 'POST');
    curl_setopt($ch, CURLOPT_POSTFIELDS, $dataString);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_HTTPHEADER, [
        'Content-Type: application/json',
        'Content-Length: ' . strlen($dataString),
    ]);
    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
    curl_setopt($ch, CURLOPT_TIMEOUT, 15);
    $result = curl_exec($ch);
    $errNo = curl_errno($ch);
    curl_close($ch);

    if ($errNo) {
        return ['success' => false, 'message' => 'خطا در اتصال به سرویس پیامک.'];
    }

    $decoded = json_decode($result, true);
    if (json_last_error() !== JSON_ERROR_NONE) {
        return ['success' => false, 'message' => 'پاسخ سرویس نامعتبر است.'];
    }

    $code = isset($decoded['code']) ? (string) $decoded['code'] : '';
    $status = isset($decoded['status']) ? trim((string) $decoded['status']) : '';
    $codeDigits = preg_replace('/\D/', '', $code);
    if ($codeDigits === '') {
        return ['success' => false, 'message' => $status !== '' ? $status : 'ارسال پیامک انجام نشد.'];
    }
    $code4 = strlen($codeDigits) >= 4 ? substr($codeDigits, -4) : $codeDigits;
    return ['success' => true, 'code' => $code4];
}
